using System.Runtime.InteropServices;

namespace SendMail {
   internal partial class Main : Form {
      readonly string fnConfig = "SendMailConfig.txt";
      readonly SendMailConfig sendMailConfig = new();
      readonly bool configLoaded = false;

      public static class NativeMethods {
         [DllImport("user32.dll", SetLastError = true)]
         [return: MarshalAs(UnmanagedType.Bool)]
         public static extern bool AddClipboardFormatListener(IntPtr hwnd);
      }

      internal Main() {
         InitializeComponent();

         string[] args = Environment.GetCommandLineArgs();
         if (args.Length > 1) {
            fnConfig = args[1];
         }

         try {
            sendMailConfig = SendMailConfig.FromFile(fnConfig);
            configLoaded = true;
         }
         catch (Exception) {
            MessageBox.Show("Konfiguarationsdatei '" + fnConfig + "' kann nicht eingelesen werden.",
               "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
         }
      }

      private void Form1_Load(object sender, EventArgs e) {
         if (!configLoaded) { // Programmabbruch: Keine Konfiguration geladen
            Close();
            return;
         }

         NativeMethods.AddClipboardFormatListener(Handle);
         string[] allAddresses;
         try {
            allAddresses = File.ReadAllText(sendMailConfig.addressListFilename).Split("\r\n");
         }
         catch (Exception) {
            MessageBox.Show("Adressenliste '" + sendMailConfig.addressListFilename + "' kann nicht eingelesen werden.",
               "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
            Close();
            return;
         }

         lblFileNames.Text = "Die Datei mit den Adressen ist:\n" + sendMailConfig.addressListFilename +
            "\n\nFehlerprotokoll: \n" + sendMailConfig.errorFilename;
         lblTestMail.Text = "Testmail an:\n" + sendMailConfig.testMailAddress;


         for (int i = 0; i < allAddresses.Length; i++) { // Erste Zeile ist keine Mail-Adresse
            string address = allAddresses[i].Trim();
            if (!string.IsNullOrEmpty(address) && !address.StartsWith('#'))
               lbToSend.Items.Add(allAddresses[i]);
         }
         lblToSendCount.Text = lbToSend.Items.Count.ToString();
         Width += 1; // damit der WebBrowser korrekt angezeigt wird.
      }

      private void cmdSend_Click(object sender, EventArgs e) {
         if (tbSubject.Text.Trim().Length == 0) {
            MessageBox.Show("Kein Betreff angegeben", "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return;
         }
         if (wbMailDisplay.DocumentText is null || wbMailDisplay.DocumentText == "") {
            MessageBox.Show("Kein Text angegeben", "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return;
         }

         DialogResult dialogResult = MessageBox.Show("Mail jetzt senden?", "Mail versenden", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2);
         if (dialogResult == DialogResult.Yes) {
            foreach (string recipient in lbToSend.Items) {
               if (recipient != "") {
                  Exception? ex = MailSender.Send(recipient, tbSubject.Text, wbMailDisplay.DocumentText, sendMailConfig);
                  if (ex != null) {
                     lbError.Items.Add(recipient);
                     lblSendError.Text = lbError.Items.Count.ToString();
                     lbError.TopIndex = lbError.Items.Count - 1; // An das Ende der listbox scrollen
                     lblSendError.Refresh();
                     lbError.Refresh();
                     String txt = DateTime.Now.ToString("G") + ": " + recipient + "### " + ex.ToString() + Environment.NewLine;
                     try {
                        File.AppendAllText(sendMailConfig.errorFilename, txt);
                     }
                     catch (Exception) {
                        MessageBox.Show("Fehler beim Schreiben in die Fehlerdatei\n" + ex.ToString(), "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
                     }
                  }
                  else {
                     lbSend.Items.Add(recipient);
                     lblSendCount.Text = lbSend.Items.Count.ToString();
                     lbSend.TopIndex = lbSend.Items.Count - 1; // An das Ende der listbox scrollen
                     lblSendCount.Refresh();
                     lbSend.Refresh();
                  }
               }
               // Frequenz herab setzen
               Thread.Sleep(10000);
            }
            MessageBox.Show("Fertig!");
         }

         else if (dialogResult == DialogResult.No) {
            // nichts tun
         }
      }

      /// <inheritdoc/>
      protected override void WndProc(ref Message m) {
         const int WM_CLIPBOARDUPDATE = 0x31D;
         if (wbMailDisplay.DocumentText is null || wbMailDisplay.DocumentText == "") {
            if (m.Msg == WM_CLIPBOARDUPDATE) {
               string clipboardData = Clipboard.GetText(TextDataFormat.Html);
               if (!string.IsNullOrEmpty(clipboardData)) {
                  int index = clipboardData.IndexOf("<html");
                  if (index >= 0) {
                     clipboardData = clipboardData.Substring(index);
                     wbMailDisplay.DocumentText = clipboardData;
                     Console.Beep();
                  }
               }
            }
         }
         base.WndProc(ref m);
      }

      private void cmdTest_Click(object sender, EventArgs e) {
         if (tbSubject.Text.Trim().Length == 0) {
            MessageBox.Show("Kein Betreff angegeben", "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return;
         }
         if (wbMailDisplay.DocumentText is null || wbMailDisplay.DocumentText == "") {
            MessageBox.Show("Kein Text angegeben", "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return;
         }

         Exception? ex = MailSender.Send(sendMailConfig.testMailAddress, tbSubject.Text, wbMailDisplay.DocumentText, sendMailConfig);
         if (ex != null)
            MessageBox.Show(ex.ToString(), "SendMail", MessageBoxButtons.OK, MessageBoxIcon.Error);
         else
            MessageBox.Show("Fertig");
      }

      private void cmdClear_Click(object sender, EventArgs e) {
         wbMailDisplay.DocumentText = null;
      }

      private void cmdRetry_Click(object sender, EventArgs e) {
         lbToSend.Items.Clear();

         foreach (String s in lbError.Items) {
            lbToSend.Items.Add(s);
         }

         lblToSendCount.Text = lbToSend.Items.Count.ToString();
         lbError.Items.Clear();
         lblSendError.Text = lbError.Items.Count.ToString();
      }

      private void cmdAbout_Click(object sender, EventArgs e) {
         SendMailAboutBox box = new SendMailAboutBox();
         box.ShowDialog();
      }
   }
}
