#pragma once

#include <Arduino.h>

// https://wiki.hshl.de/wiki/index.php/Schrittmotor_28BYJ-48_mit_ULN2003_Treiberplatine
// https://espressif-docs.readthedocs-hosted.com/projects/arduino-esp32/en/latest/api/timer.html

// Forward declaration of the timer interrupt handler
void ARDUINO_ISR_ATTR onMotorTimer(void* arg);

// Motor types
enum class MotorTypes {
   Simple, // Single coil activation
   Strong  // Double coil activation for more torque, uses more power
};

// Rotation direction
enum class Direction {
   CW, // Clockwise, Rechts herum
   CCW // Counter-clockwise, Links herum
};

class M28BYJ_48 {
   friend void ARDUINO_ISR_ATTR onMotorTimer(void* arg);

public:
   // Constructor with motor control pins and type
   // pin1-pin4: Motor control pins
   // type: Motor type (Simple or Strong)
   // gearClearance: Clearance between direction changes (in steps)
   M28BYJ_48(uint8_t pin1, uint8_t pin2, uint8_t pin3, uint8_t pin4, uint8_t gearClearance, MotorTypes type = MotorTypes::Simple);

   // Initialize the motor pins and timer
   void begin();
   // Deinitialize the motor and timer
   void end();

   // Set the rotation direction
   // Applies clearance steps if changing direction
   void setDirection(Direction dir);

   // Set the gear clearance (in steps)
   void setGearClearance(uint8_t clearance);


   // Step the motor one step in the specified direction
   // This method does the job
   void step(Direction dir);

   // Step the motor one step in the current direction
   void step() { step(dir); }

   // Step the motor one step clockwise
   void stepCW() { step(Direction::CW); }
   // Step the motor one step counter-clockwise
   void stepCCW() { step(Direction::CCW); }

   // Run the motor in the specified direction at the given speed (steps per second)
   void run(Direction dir, uint16_t stepsPerSecond);
   // Run the motor in the current direction at the given speed (steps per second)
   void run(uint16_t stepsPerSecond) { run(dir, stepsPerSecond); }

   // Run the motor in the specified direction for a given number of steps at the given speed (steps per second)
   // steps: Number of steps to run, or -1 for unlimited
   void run(Direction dir, int steps, uint16_t stepsPerSecond);

   // Run the motor in the current direction for a given number of steps at the given speed (steps per second)
   void run(int steps, uint16_t stepsPerSecond) { run(dir, steps, stepsPerSecond); }

   // Stop the motor.
   void stop();

   // Check if the motor is currently running
   bool isRunning() const {
      return isTimerRunning;
   }

   // Check if the motor has been initialized
   bool isInitialized() const {
      return isMotorInitialized;
   }

protected:
   uint8_t pin1; // Motor control pins
   uint8_t pin2;
   uint8_t pin3;
   uint8_t pin4;
   uint8_t clearance; // Clearance between direction changes (in steps)

   // Current step index (0-3)
   uint8_t stepIndex = 0;

   // For handling steps in commands
   // -1: unlimited steps
   int remainingSteps = -1;

   // Current rotation direction
   Direction dir = Direction::CCW;

   // Hardware timer for controlling motor speed
   hw_timer_t* motorTimer = NULL;

   // Flag to indicate if the timer is running
   volatile bool isTimerRunning = false;

   // Flag to indicate if the motor has been initialized
   volatile bool isMotorInitialized = false;

   // Pointer to the function that sets the motor pins
   void (M28BYJ_48::* setPin)(uint8_t) const;

   // Set the motor pins for single coil activation
   void setSinglePin(uint8_t stepIndex) const;

   // Set the motor pins for double coil activation
   void setDoublePin(uint8_t stepIndex) const;

   // Step the motor one step in the current direction
   void internalStep(Direction dir);

   // Step the motor one step clockwise
   void internalStepCW();
   // Step the motor one step counter-clockwise
   void internalStepCCW();

   // Run the motor in the specified direction for a given number of steps at the given speed (steps per second)
   // Stop any ongoing movement before starting
   void internalRun(Direction dir, int steps, uint16_t stepsPerSecond);
};


